﻿using System;
using System.Diagnostics;
using System.Net;
using gov.va.med.VBECS.Communication.Channels;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Protocols;

namespace gov.va.med.VBECS.Communication.Server
{
    /// <summary>
    /// Listens for incoming messages (by using internal communication channel).
    /// Sends ping requests to the server.
    /// </summary>
    internal class ClientDummy : IClientDummy
    {
        private readonly IChannel _internalChannel;

        public ClientDummy(long theId, IChannel theChannel)
        {
            Id = theId;
            _internalChannel = theChannel;
            _internalChannel.MessageReceived += internal_channel_message_received;
            _internalChannel.MessageSent += internal_channel_message_sent;
            _internalChannel.Disconnected += internal_channel_disconnected;
        }

        #region IClientDummy Members

        public event EventHandler Disconnected;

        public event EventHandler<MessageEventArgs> MessageReceived;

        public event EventHandler<MessageEventArgs> MessageSent;

        public DateTime LastReceivedTime { get; protected set; }

        public DateTime LastSentTime { get; protected set; }

        public IProtocol Protocol
        {
            get { return _internalChannel.Protocol; }
            set { _internalChannel.Protocol = value; }
        }

        public long Id { get; private set; }

        public IPEndPoint EndPoint
        {
            get { return _internalChannel.EndPoint; }
        }

        public CommunicationStatus Status { get; protected set; }

        public void Disconnect()
        {
            if (null != _internalChannel)
            {
                _internalChannel.Disconnect();
            }
        }

        public void Send(IMessage aMessage)
        {
            if (null != _internalChannel)
            {
                _internalChannel.Send(aMessage);
            }
        }

        #endregion

        private void internal_channel_disconnected(object sender, EventArgs e)
        {
            FireDisconnectedEvent();
        }

        private void internal_channel_message_sent(object sender, MessageEventArgs e)
        {
            FireMessageSentEvent(e.Message);
        }

        private void internal_channel_message_received(object sender, MessageEventArgs e)
        {
            var theMessage = e.Message;
            //Note: if ping message doesn't derived from PingMessage it will proceed to MessageReceived event
            if (theMessage is PingMessage)
            {
#if TRACE
                Trace.WriteLine("Ping message received from client #" + Id);
#endif
                _internalChannel.Send(new PingMessage(theMessage.Id));
                return;
            }
            FireMessageReceivedEvent(theMessage);
        }

        protected virtual void FireDisconnectedEvent()
        {
            if (null != Disconnected)
            {
                Disconnected.Invoke(this, EventArgs.Empty);
            }
        }

        protected virtual void FireMessageReceivedEvent(IMessage theMessage)
        {
            if (null != MessageReceived)
            {
                MessageReceived.Invoke(this, new MessageEventArgs(theMessage));
            }
        }

        protected virtual void FireMessageSentEvent(IMessage theMessage)
        {
            if (null != MessageSent)
            {
                MessageSent.Invoke(this, new MessageEventArgs(theMessage));
            }
        }
    }
}